"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const runtime_core_1 = require("@postman/runtime.core");
const strip_json_comments_1 = __importDefault(require("strip-json-comments"));
const file_type_1 = __importDefault(require("file-type"));
const parse_message_1 = require("./utils/parse-message");
const data_transformers_1 = require("./utils/data-transformers");
const mime_types_1 = require("./utils/mime-types");
const validators_1 = require("./utils/validators");
const parse_url_1 = require("./utils/parse-url");
exports.default = (async function handler(item, agent, context) {
    if (!item.payload.url) {
        throw new Error('URL is required');
    }
    const url = (0, parse_url_1.parseUrl)(item.payload.url);
    const { variables } = context;
    const settings = {
        connectOptions: {
            handshakeTimeout: item.payload.settings?.handshakeTimeout ?? 0,
            maxPayload: (item.payload.settings?.maxPayload ?? 10) * 1024 * 1024,
            followRedirects: true,
        },
        reconnectOptions: {
            retryCount: item.payload.settings?.retryCount ?? 0,
            retryDelay: item.payload.settings?.retryDelay ?? 5000,
        },
    };
    const headers = (0, data_transformers_1.transformKVItemToObject)(item.payload.headers || []);
    const protocols = (0, data_transformers_1.prepareProtocols)(headers);
    const tlsOptions = url.isTLS ?
        {
            rejectUnauthorized: Boolean(item.payload.settings?.strictSSL),
            secureContext: typeof context.secureContext === 'function' ?
                await context.secureContext(item.payload.url)
                : context.secureContext,
        }
        : {};
    const connection = await agent.connect({
        url: url.url,
        headers,
        protocols,
        ...settings,
        tlsOptions,
    });
    const events = runtime_core_1.EventChannel.specific(context.events);
    const encoder = new TextEncoder();
    const onPublish = (event) => {
        let payload = event.payload.payload;
        if (variables) {
            payload = variables.replaceIn(payload);
        }
        if (event.payload.messageType === mime_types_1.MessageType.POSTMAN_JSON) {
            payload = (0, strip_json_comments_1.default)(payload, { whitespace: false });
        }
        else if (event.payload.messageType === mime_types_1.MessageType.BINARY) {
            payload = (0, parse_message_1.binaryStringToUint8Array)(payload, event.payload.messageSubType);
        }
        const mimeType = mime_types_1.MIME_TYPE_MAP[event.payload.messageType];
        connection.send(payload);
        // Generate synthetic sent message event
        let sentPayload;
        let messageType;
        let payloadSize;
        if (typeof payload === 'string') {
            sentPayload = payload;
            payloadSize = encoder.encode(payload).byteLength;
            messageType = 'String';
        }
        else {
            sentPayload = (0, parse_message_1.uint8ArrayToString)(payload);
            payloadSize = payload.byteLength;
            messageType = 'Uint8Array';
        }
        events.emit('sent-message', {
            data: { payload: sentPayload, type: messageType },
            mimeType,
            size: payloadSize,
        });
    };
    const onClose = (event) => {
        connection.close(event.payload.code, event.payload.reason);
    };
    return new Promise((resolve) => {
        const onDone = () => {
            connection.close();
            events.off('close', onClose).off('publish', onPublish);
            resolve();
        };
        events.on('publish', onPublish).on('close', onClose).onCleanup(onDone);
        connection
            .on('open', ({ request, response }) => {
            const payload = {};
            if (request) {
                const { href, httpVersion, ...rest } = request;
                payload.handshakeRequest = { url: href, ...rest };
            }
            if (response) {
                const { httpVersion, ...rest } = response;
                payload.handshakeResponse = rest;
            }
            events.emit('connected', payload);
        })
            .on('reconnect', () => {
            events.emit('reconnecting', {});
        })
            .on('error', ({ handshakeRequest: request, handshakeResponse: response, error }) => {
            const payload = {
                error: { message: error.message },
            };
            if (request) {
                const { href, httpVersion, ...rest } = request;
                payload.handshakeRequest = { url: href, ...rest };
            }
            if (response) {
                const { httpVersion, ...rest } = response;
                payload.handshakeResponse = rest;
            }
            events.emit('error', payload);
        })
            .on('end', ({ code, reason, aborted }) => {
            if (aborted) {
                events.emit('aborted', {});
            }
            else {
                events.emit('disconnected', { code, reason });
            }
            onDone();
        })
            .on('message', async (event) => {
            const meta = {
                mimeType: 'text/plain',
                size: 0,
                fileExtension: '',
            };
            if (typeof event.message === 'string') {
                meta.size = encoder.encode(event.message).byteLength;
                (0, validators_1.isJSON)(event.message) && (meta.mimeType = 'application/json');
                events.emit('received-message', {
                    data: {
                        payload: event.message,
                        type: 'String',
                    },
                    ...meta,
                });
                return;
            }
            else {
                meta.size = event.message.byteLength;
                try {
                    const result = await file_type_1.default.fromBuffer(event.message);
                    meta.mimeType = result?.mime || 'application/octet-stream';
                    meta.fileExtension = result?.ext || 'bin';
                }
                catch (err) {
                    meta.mimeType = 'application/octet-stream';
                    meta.fileExtension = 'bin';
                }
                finally {
                    events.emit('received-message', {
                        data: {
                            payload: (0, parse_message_1.uint8ArrayToString)(event.message),
                            type: 'Uint8Array',
                        },
                        ...meta,
                    });
                }
            }
        });
    });
});
//# sourceMappingURL=handler.js.map