"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasNonPrintableCharacters = exports.getContentType = void 0;
const file_type_1 = __importDefault(require("file-type"));
async function getContentType(packet) {
    const { payload, properties = undefined } = packet;
    if (properties) {
        // Not using content type from properties because it could be any string.
        const { payloadFormatIndicator = undefined } = properties;
        if (payloadFormatIndicator === true) {
            return 'text/plain';
        }
    }
    let contentType;
    let payloadString = payload;
    if (typeof payloadString !== 'string') {
        try {
            const fileType = await file_type_1.default.fromBuffer(payloadString);
            contentType = fileType?.mime;
            if (contentType) {
                return contentType;
            }
            else if (hasNonPrintableCharacters(payload)) {
                return 'application/octet-stream';
            }
            else {
                payloadString = payload.toString('utf8');
            }
        }
        catch (error) {
            throw new Error(`Unable to detect content type: ${error}`);
        }
    }
    contentType =
        isMessageObject(payloadString) ? 'application/json' : 'text/plain';
    return contentType;
}
exports.getContentType = getContentType;
/**
 * Check if value is non-null + non-Array object
 */
function isObject(value) {
    return value !== null && typeof value === 'object' && !Array.isArray(value);
}
/**
 * Checks if a value is a JSON object.
 */
function isMessageObject(value) {
    let parsed;
    try {
        parsed = JSON.parse(value);
    }
    catch (_) {
        return false;
    }
    if (!isObject(parsed))
        return false;
    return true;
}
/**
 * A function to check if a payload has non-printable ASCII characters and should be displayed as binary.
 */
function hasNonPrintableCharacters(payload) {
    if (typeof payload === 'string') {
        return /[\u0000-\u0009\u000B-\u000C\u000E-\u001F\u007F-\u009F\uFFFD]/.test(payload);
    }
    else {
        const uint8Array = new Uint8Array(payload);
        return uint8Array.some((byte) => {
            // ASCII values for line breaks are 10 (LF) and 13 (CR)
            if (byte === 10 || byte === 13) {
                return false;
            }
            return byte < 32 || byte > 126;
        });
    }
}
exports.hasNonPrintableCharacters = hasNonPrintableCharacters;
//# sourceMappingURL=content-type.js.map