"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const protobufjs_1 = __importDefault(require("@postman/protobufjs"));
const js_base64_1 = require("js-base64");
// Generates a JSON Schema that describes the given Protobuf.Type.
function generateJSONSchema(rootType) {
    const definitions = {};
    const jsonPointers = new Map();
    const requiredByUser = new Set();
    const mapFieldPaths = [];
    const currFieldPath = [];
    return {
        $schema: 'http://json-schema.org/draft-07/schema',
        definitions,
        ...declareSchema(rootType),
        // caching the paths to map fields so that we can
        // efficiently sanitize the payload at runtime
        mapFields: mapFieldPaths,
    };
    // Returns a (memoized) subschema that describes the given Type or Enum.
    function declareSchema(type) {
        let jsonPointer = jsonPointers.get(type);
        if (!jsonPointer) {
            jsonPointer = `#/definitions/${type.fullName}`;
            jsonPointers.set(type, jsonPointer);
            definitions[type.fullName] = generateSchema(type);
        }
        return { $ref: jsonPointer };
    }
    // Generates a new (not memoized) subschema from the given Type or Enum.
    function generateSchema(type) {
        if (type instanceof protobufjs_1.default.Enum) {
            const enumValues = [
                ...Object.keys(type.values), // Enum names (strings)
                ...Object.values(type.values), // Enum ids (integers)
                null,
            ];
            return {
                enum: Array.from(new Set(enumValues)),
                title: `enum ${type.name}`,
                description: `${Object.entries(type.values)
                    .map(([k, v]) => `${JSON.stringify(k)} (${v})`)
                    .join(' | ')}`,
                default: type.valuesById[0] || 0,
            };
        }
        if (type instanceof protobufjs_1.default.Type) {
            let isEmpty = true;
            const properties = {};
            const required = [];
            for (const field of Object.values(type.fields)) {
                isEmpty = false;
                currFieldPath.push(field.name);
                applyField(field, properties, required);
                currFieldPath.pop();
            }
            if (!isEmpty) {
                // Only non-empty message types are actually required. Everything else has a default value.
                requiredByUser.add(type);
            }
            return {
                type: isEmpty ? ['object', 'null'] : 'object',
                properties,
                required,
                additionalProperties: { title: 'Unknown field' },
                title: `message ${type.name}`,
                description: `{ ${Object.keys(type.fields).join(', ')} }`,
                default: {},
            };
        }
        console.warn(`Unrecognized Protobuf class: ${type && type.constructor.name}`);
        return {};
    }
    // Augments a "properties" object and "required" array with a given field.
    function applyField(field, properties, required) {
        let schema;
        if (field.resolvedType) {
            schema = declareSchema(field.resolvedType);
        }
        else {
            schema = getKnownSchema(field.type);
            schema.title = field.type;
            schema.default = normalizeProtobufValue(field.typeDefault);
        }
        if (field.map) {
            const keyType = field.keyType;
            schema = {
                oneOf: [
                    // JSON representation
                    {
                        type: ['object', 'null'],
                        additionalProperties: schema,
                        title: `map<${keyType}, ${field.type}>`,
                        default: {},
                    },
                    // Array representation for backward compatibility
                    // Due to how we previously decoded map fields, we need to support this
                    {
                        type: ['array', 'null'],
                        items: {
                            type: 'object',
                            properties: {
                                key: (() => {
                                    const keySchema = getKnownSchema(keyType);
                                    keySchema.title = keyType;
                                    return keySchema;
                                })(),
                                value: schema,
                            },
                            required: ['key', 'value'],
                            additionalProperties: false,
                        },
                        title: `repeated { "key: ${keyType}, "value": ${field.type} }`,
                        default: [],
                    },
                ],
                title: `map<${keyType}, ${field.type}>`,
                default: {},
            };
            mapFieldPaths.push(currFieldPath.join('.'));
        }
        else if (field.repeated) {
            schema = {
                type: ['array', 'null'],
                items: schema,
                title: `repeated ${field.type}`,
                default: [],
            };
        }
        else if (field.required) {
            if (requiredByUser.has(field.resolvedType)) {
                required.push(field.name);
            }
        }
        properties[field.name] = schema;
    }
}
exports.default = generateJSONSchema;
// Returns the subschema of a well-known Protobuf type.
function getKnownSchema(typeName) {
    switch (typeName) {
        case 'bool':
            return {
                type: ['boolean', 'null'],
            };
        case 'string':
            return {
                type: ['string', 'null'],
            };
        case 'bytes':
            return {
                type: ['string', 'array', 'null'],
                pattern: '^([a-zA-Z0-9+/]{4})*([a-zA-Z0-9+/]{2}(==)?|[a-zA-Z0-9+/]{3}=?)?$',
                items: {
                    type: 'integer',
                    default: 0,
                    minimum: 0,
                    maximum: 255,
                },
                description: 'A string of base64, or an array of numbers between 0 and 255',
            };
        case 'int32':
        case 'sint32':
        case 'sfixed32':
            return {
                type: ['integer', 'null'],
                minimum: -0x80000000,
                maximum: 0x7fffffff,
                description: 'An integer between -2147483648 and 2147483647',
            };
        case 'uint32':
        case 'fixed32':
            return {
                type: ['integer', 'null'],
                minimum: 0,
                maximum: 0xffffffff,
                description: 'An integer between 0 and 4294967295',
            };
        case 'int64':
        case 'sint64':
        case 'sfixed64':
            return {
                type: ['integer', 'string', 'null'],
                minimum: Number(0n - 2n ** 63n), // This is not precise, due to JavaScript
                maximum: Number(2n ** 63n - 1n), // This is not precise, due to JavaScript
                pattern: '^[0-9]+$', // Can't enforce numeric limits on strings
                description: 'An integer between -9223372036854775808 and 9223372036854775807',
            };
        case 'uint64':
        case 'fixed64':
            return {
                type: ['integer', 'string', 'null'],
                minimum: 0,
                maximum: Number(2n ** 64n - 1n), // This is not precise, due to JavaScript
                pattern: '^[0-9]+$', // Can't enforce numeric limits on strings
                description: 'An integer between 0 and 18446744073709551615',
            };
        case 'float':
        case 'double':
            return {
                anyOf: [
                    { type: ['number', 'null'] },
                    { type: 'string', enum: ['NaN', 'Infinity', '-Infinity'] },
                ],
                description: 'A number, or one of the strings "NaN", "Infinity", or "-Infinity"',
            };
        default:
            console.warn(`Unrecognized Protobuf type: ${typeName}`);
            return {};
    }
}
// Convert a Protobuf runtime value into a canonical JSON value.
function normalizeProtobufValue(value) {
    if (protobufjs_1.default.util.Long.isLong(value)) {
        return value.toString(10);
    }
    if (value instanceof Uint8Array) {
        return js_base64_1.Base64.fromUint8Array(value);
    }
    if (Array.isArray(value)) {
        return js_base64_1.Base64.fromUint8Array(new Uint8Array(value));
    }
    return value;
}
//# sourceMappingURL=generate-json-schema.js.map